import torch
import torch.nn as nn
import torch.nn.functional as F

from einops import rearrange

from .perceptual_loss import LPIPS
# from taming.modules.losses.vitper import ViTPER 
# from taming.modules.losses.clip_encoder import CLIP_RN50

from .discriminator import NLayerDiscriminator, weights_init

import pdb

class DummyLoss(nn.Module):
    def __init__(self):
        super().__init__()


def adopt_weight(weight, global_step, threshold=0, value=0.):
    if global_step < threshold:
        weight = value
    return weight


def hinge_d_loss(logits_real, logits_fake):
    loss_real = torch.mean(F.relu(1. - logits_real))
    loss_fake = torch.mean(F.relu(1. + logits_fake))
    d_loss = 0.5 * (loss_real + loss_fake)
    return d_loss


def vanilla_d_loss(logits_real, logits_fake):
    d_loss = 0.5 * (
        torch.mean(torch.nn.functional.softplus(-logits_real)) +
        torch.mean(torch.nn.functional.softplus(logits_fake)))
    return d_loss


class VQLPIPSWithDiscriminator(nn.Module):
    def __init__(self, 
                disc_start, 
                codebook_weight=1.0, 
                pixelloss_weight=1.0,
                disc_num_layers=3, 
                disc_in_channels=3, 
                disc_factor=1.0, 
                disc_weight=1.0,
                perceptual_weight=1.0, 
                perceptual_type='vgg',
                logitlaplace_loss_weight=0., 
                use_actnorm=False, 
                disc_conditional=False, 
                disc_ndf=64, 
                disc_loss="hinge",
                rec_loss_type='l1'):
        super().__init__()
        assert disc_loss in ["hinge", "vanilla"]
        self.codebook_weight = codebook_weight
        self.pixel_weight = pixelloss_weight
        
        print(f"Using {perceptual_type} to calculate perceptual loss")
        if perceptual_type == 'vgg': 
            self.perceptual_loss = LPIPS().eval() if perceptual_weight > 0 else None
        elif perceptual_type == 'clip_vit':
            self.perceptual_loss = CLIP_ViT().eval() if perceptual_weight > 0 else None
        elif perceptual_type == 'clip_r50':
            self.perceptual_loss = CLIP_RN50().eval() if perceptual_weight > 0 else None
        else:
            raise NotImplementedError
        print(f"reconstuction loss type: {rec_loss_type}")
        self.rec_loss_type = rec_loss_type
        print(f"use logitlaplace_loss_weight with the weight of {logitlaplace_loss_weight}")
        self.logitlaplace_loss_weight = logitlaplace_loss_weight
        
        self.perceptual_weight = perceptual_weight
        
        self.discriminator_weight = disc_weight

        if self.discriminator_weight > 0:
            self.discriminator = NLayerDiscriminator(input_nc=disc_in_channels,
                                                 n_layers=disc_num_layers,
                                                 use_actnorm=use_actnorm,
                                                 ndf=disc_ndf
                                                 ).apply(weights_init)
        else:
            self.discriminator = None

        self.discriminator_iter_start = disc_start
        if disc_loss == "hinge":
            self.disc_loss = hinge_d_loss
        elif disc_loss == "vanilla":
            self.disc_loss = vanilla_d_loss
        else:
            raise ValueError(f"Unknown GAN loss '{disc_loss}'.")
        print(f"VQLPIPSWithDiscriminator running with {disc_loss} loss.")
        self.disc_factor = disc_factor
        self.disc_conditional = disc_conditional

    def calculate_adaptive_weight(self, nll_loss, g_loss, last_layer=None):
        if last_layer is not None:
            nll_grads = torch.autograd.grad(nll_loss, last_layer, retain_graph=True)[0]
            g_grads = torch.autograd.grad(g_loss, last_layer, retain_graph=True)[0]
        else:
            nll_grads = torch.autograd.grad(nll_loss, self.last_layer[0], retain_graph=True)[0]
            g_grads = torch.autograd.grad(g_loss, self.last_layer[0], retain_graph=True)[0]

        d_weight = torch.norm(nll_grads) / (torch.norm(g_grads) + 1e-4)
        d_weight = torch.clamp(d_weight, 0.0, 1e4).detach()
        d_weight = d_weight * self.discriminator_weight
        return d_weight

    def forward(self,
                codebook_loss,
                inputs,
                reconstructions_bias,
                global_step,
                optimizer_idx=0,
                last_layer=None, 
                cond=None, 
                split="train",
                norm_target=False,
                patch_size=16, 
                img_h=224, 
                img_w=224,
                rec_out_channels=3):
        
        def reshape_token_to_map(tokens):
            return rearrange(tokens, 'b (h w) (p1 p2) c -> b c (h p1) (w p2)', p1=patch_size, p2=patch_size, h=img_h//patch_size, w=img_w//patch_size)
            
        if len(reconstructions_bias.shape) == 3:
            B, N, E = reconstructions_bias.shape
            if rec_out_channels == 3:
                reconstructions = reconstructions_bias
            else:
                reconstructions = reconstructions_bias[:, :, :E // 2].sigmoid()
                bias = torch.exp(reconstructions_bias[:, :, E // 2:])
        else:
            reconstructions = reconstructions_bias
            
        if norm_target:
            # TODO: take it into logit-pixel space to calculate l2 loss
            images_squeeze = rearrange(inputs, 'b c (h p1) (w p2) -> b (h w) (p1 p2) c', p1=patch_size, p2=patch_size)
            images_norm = (images_squeeze - images_squeeze.mean(dim=-2, keepdim=True)
                    ) / (images_squeeze.var(dim=-2, unbiased=True, keepdim=True).sqrt() + 1e-6)
            norm_patch_target = rearrange(images_norm, 'b n p c -> b n (p c)')
            if self.rec_loss_type == 'l1':
                rec_loss = torch.abs(norm_patch_target.contiguous() - reconstructions.contiguous())
            elif self.rec_loss_type == 'smooth-l1':
                rec_loss = F.smooth_l1_loss(reconstructions.contiguous(), norm_patch_target.contiguous(), reduction='none')
            elif self.rec_loss_type == 'l2':
                rec_loss = F.mse_loss(norm_patch_target.contiguous(), reconstructions.contiguous(), reduction='none')
            else:
                raise NotImplementedError
            # reshape the rec loss for match the dim with others loss
            rec_loss = rearrange(rec_loss, 'b n (p c) -> b n p c', c=3)
            rec_loss = reshape_token_to_map(rec_loss)
            
            # pdb.set_trace()
            reconstructions = rearrange(reconstructions, 'b n (p c) -> b n p c', c=3)
            reconstructions = reconstructions * (images_squeeze.var(dim=-2, unbiased=True, keepdim=True).sqrt() + 1e-6) + images_squeeze.mean(dim=-2, keepdim=True)
            # reconstructions = rearrange(reconstructions, 'b (h w) (p1 p2) c -> b c (h p1) (w p2)', p1=patch_size, p2=patch_size, h=img_h//patch_size, w=img_w//patch_size)
            reconstructions = reshape_token_to_map(reconstructions)
        else:
            # pdb.set_trace()
            if inputs.shape != reconstructions.shape:
                reconstructions = rearrange(reconstructions, 'b n (p c) -> b n p c', c=3)
                # reconstructions = rearrange(reconstructions, 'b (h w) (p1 p2) c -> b c (h p1) (w p2)', p1=patch_size, p2=patch_size, h=img_h//patch_size, w=img_w//patch_size)
                reconstructions = reshape_token_to_map(reconstructions)
                
            if self.rec_loss_type == 'l1':
                rec_loss = torch.abs(inputs.contiguous() - reconstructions.contiguous())
            elif self.rec_loss_type == 'smooth-l1':
                rec_loss = F.smooth_l1_loss(reconstructions.contiguous(), inputs.contiguous(), reduction='none')
            elif self.rec_loss_type == 'l2':
                rec_loss = F.mse_loss(inputs.contiguous(), reconstructions.contiguous(), reduction='none')
            else:
                raise NotImplementedError
        
        if self.logitlaplace_loss_weight > 0:
            bias = rearrange(bias, 'b n (p c) -> b n p c', c=3)
            bias = reshape_token_to_map(bias)
            # bias = rearrange(bias, 'b (h w) (p1 p2) c -> b c (h p1) (w p2)', p1=patch_size, p2=patch_size, h=img_h//patch_size, w=img_w//patch_size)
            # pdb.set_trace()
            reconstructions_logit = rearrange(reconstructions_bias[:, :, :E // 2], 'b n (p c) -> b n p c', c=3)
            reconstructions_logit = reshape_token_to_map(reconstructions_logit)
            loglap_loss = (torch.log(2 * bias * inputs * (1 - inputs)) + torch.abs(torch.logit(inputs) - reconstructions_logit) / bias) * self.logitlaplace_loss_weight
        else:
            loglap_loss = torch.zeros_like(rec_loss, requires_grad=True).to(rec_loss)

        if self.perceptual_weight > 0:
            p_loss = self.perceptual_loss(inputs.contiguous(), reconstructions.contiguous()) * self.perceptual_weight
        else:
            p_loss = torch.tensor([0.0], requires_grad=True).to(reconstructions)

        # pdb.set_trace()
        nll_loss = rec_loss + p_loss + loglap_loss
        #nll_loss = torch.sum(nll_loss) / nll_loss.shape[0]
        nll_loss = torch.mean(nll_loss)

        # now the GAN part
        if optimizer_idx == 0:
            # generator update
            if self.discriminator is None:
                loss = nll_loss + self.codebook_weight * codebook_loss.mean()

                log = {"{}/total_loss".format(split): loss.clone().detach().mean(),
                   "{}/quant_loss".format(split): codebook_loss.detach().mean(),
                   "{}/nll_loss".format(split): nll_loss.detach().mean(),
                   "{}/rec_loss".format(split): rec_loss.detach().mean(),
                   "{}/loglap_loss".format(split): loglap_loss.detach().mean(),
                   "{}/p_loss".format(split): p_loss.detach().mean(),
                   }
                return loss, log
                
            else:
                if cond is None:
                    assert not self.disc_conditional
                    logits_fake = self.discriminator(reconstructions.contiguous())
                else:
                    assert self.disc_conditional
                    logits_fake = self.discriminator(torch.cat((reconstructions.contiguous(), cond), dim=1))
                g_loss = -torch.mean(logits_fake)

                try:
                    d_weight = self.calculate_adaptive_weight(nll_loss, g_loss, last_layer=last_layer)
                except RuntimeError:
                    assert not self.training
                    d_weight = torch.tensor(0.0)

                disc_factor = adopt_weight(self.disc_factor, global_step, threshold=self.discriminator_iter_start)
                loss = nll_loss + d_weight * disc_factor * g_loss + self.codebook_weight * codebook_loss.mean()

                log = {"{}/total_loss".format(split): loss.clone().detach().mean(),
                    "{}/quant_loss".format(split): codebook_loss.detach().mean(),
                    "{}/nll_loss".format(split): nll_loss.detach().mean(),
                    "{}/rec_loss".format(split): rec_loss.detach().mean(),
                    "{}/p_loss".format(split): p_loss.detach().mean(),
                    "{}/d_weight".format(split): d_weight.detach(),
                    "{}/disc_factor".format(split): torch.tensor(disc_factor),
                    "{}/g_loss".format(split): g_loss.detach().mean(),
                    }
                return loss, log

        if optimizer_idx == 1:
            assert self.discriminator is not None
            # second pass for discriminator update
            if cond is None:
                logits_real = self.discriminator(inputs.contiguous().detach())
                logits_fake = self.discriminator(reconstructions.contiguous().detach())
            else:
                logits_real = self.discriminator(torch.cat((inputs.contiguous().detach(), cond), dim=1))
                logits_fake = self.discriminator(torch.cat((reconstructions.contiguous().detach(), cond), dim=1))

            disc_factor = adopt_weight(self.disc_factor, global_step, threshold=self.discriminator_iter_start)
            d_loss = disc_factor * self.disc_loss(logits_real, logits_fake)

            log = {"{}/disc_loss".format(split): d_loss.clone().detach().mean(),
                   "{}/logits_real".format(split): logits_real.detach().mean(),
                   "{}/logits_fake".format(split): logits_fake.detach().mean()
                   }
            return d_loss, log
